<?php

/**
 * @author The Courier Guy
 * @package tcg/core
 * @version 1.0.1
 */
class ParcelPerfectApiPayload
{
    public $globalFactor = 50;

    /**
     * ParcelPerfectApiPayload constructor.
     */
    public function __construct()
    {
    }

    /**
     * @param array $parameters
     *
     * @return array
     */
    private function getOriginPayload($parameters)
    {
        return [
            'accnum'         => isset($parameters['account']) ? $parameters['account'] : '',
            'reference'      => '',
            'origperadd1'    => isset($parameters['shopAddress1']) ? $parameters['shopAddress1'] : '',
            'origperadd2'    => isset($parameters['shopAddress2']) ? $parameters['shopAddress2'] : '',
            'origperadd3'    => isset($parameters['shopCity']) ? $parameters['shopCity'] : '',
            'origperadd4'    => '',
            'origperphone'   => isset($parameters['shopPhone']) ? $parameters['shopPhone'] : '',
            'origpercell'    => '',
            'origplace'      => isset($parameters['shopArea']) ? $parameters['shopArea'] : '',
            'origtown'       => isset($parameters['shopPlace']) ? $parameters['shopPlace'] : '',
            'origpers'       => isset($parameters['company_name']) ? $parameters['company_name'] : '',
            'origpercontact' => isset($parameters['contact_name']) ? $parameters['contact_name'] : '',
            'origperpcode'   => isset($parameters['shopPostalCode']) ? $parameters['shopPostalCode'] : '',
            'notifyorigpers' => 1,
            'origperemail'   => isset($parameters['shopEmail']) ? $parameters['shopEmail'] : '',
        ];
    }

    /**
     * @param array $package
     *
     * @return array
     */
    private function getDestinationPayloadForQuote($package)
    {
        global $TCG_Plugin;
        $customer                 = WC()->customer;
        $customShippingProperties = $TCG_Plugin->getShippingCustomProperties();
        $address1                 = $customer->get_billing_address_1();
        $address2                 = $customer->get_billing_address_2();
        $city                     = $customer->get_billing_city();
        if (!empty($package['destination'])) {
            $destination = $package['destination'];
            if (!empty($package['destination'])) {
                $address1 = $destination['address'];
                $address2 = $destination['address_2'];
                $city     = $destination['city'];
            }
        }
        $postCode = '';
        if (!empty($package['postcode'])) {
            $postCode = $package['postcode'];
        }

        return [
            'destperadd1'    => $address1,
            'destperadd2'    => $address2,
            'destperadd3'    => $city,
            'destperadd4'    => '',
            'destperphone'   => $customer->get_billing_phone(),
            'destpercell'    => $customer->get_billing_phone(),
            'destplace'      => $customShippingProperties['tcg_place_id'],
            'desttown'       => $customShippingProperties['tcg_place_label'],
            'destpers'       => $customer->get_billing_company(),
            'destpercontact' => $customer->get_shipping_first_name() . ' ' . $customer->get_shipping_last_name(),
            'destperpcode'   => $postCode,
            'notifydestpers' => 1,
            'destperemail'   => $destination['email'] ?? '',
        ];
    }

    /**
     * @param WC_Order $order
     *
     * @return array
     */
    private function getDestinationPayloadForCollection($order)
    {
        return [
            'destperadd1'    => $order->get_shipping_address_1(),
            'destperadd2'    => $order->get_shipping_address_2(),
            'destperadd3'    => $order->get_shipping_company(),
            'destperadd4'    => '',
            'destperphone'   => $order->get_billing_phone(),
            'destpercell'    => $customer->get_billing_phone(),
            'destplace'      => $order->shipping_area,
            'desttown'       => $order->shipping_place,
            'destpers'       => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
            'destpercontact' => $order->get_billing_phone(),
            'destperpcode'   => $order->get_shipping_postcode(),
            'notifydestpers' => 1,
            'destperemail'   => $order->get_billing_email(),
            'reference'      => $order->get_id(),
        ];
    }

    /**
     * @return array
     */
    private function getInsurancePayloadForQuote()
    {
        global $TCG_Plugin;
        $result                   = [];
        $customShippingProperties = $TCG_Plugin->getShippingCustomProperties();
        $insurance                = $customShippingProperties['tcg_insurance'];
        if ($insurance) {
            $result = [
                'insuranceflag' => 1,
                'declaredvalue' => WC()->cart->get_displayed_subtotal(),
            ];
        }

        return $result;
    }

    /**
     * @param WC_Order $order
     *
     * @return array
     */
    private function getInsurancePayloadForCollection($order)
    {
        $result = [];
        if (get_post_meta($order->get_id(), '_billing_insurance', true) || get_post_meta(
                $order->get_id(),
                '_shipping_insurance',
                true
            )) {
            $result = [
                'insuranceflag' => 1,
            ];
        }

        return $result;
    }

    /**
     * @param array $parameters
     * @param array $items
     *
     * @return array
     */
    private function getContentsPayload($parameters, $items)
    {
        $waybillDescriptionOverride = isset($parameters['remove_waybill_description']) ? $parameters['remove_waybill_description'] === 'yes' : false;
        $r1                         = $r2 = [];

        /** Get the standard parcel sizes
         * At least one must be set or default to standard size
         */
        $globalParcels = [];
        for ($i = 1; $i < 4; $i++) {
            $globalParcel              = [];
            $product_length_per_parcel = isset($parameters['product_length_per_parcel_' . $i]) ? $parameters['product_length_per_parcel_' . $i] : '';
            $product_width_per_parcel  = isset($parameters['product_width_per_parcel_' . $i]) ? $parameters['product_width_per_parcel_' . $i] : '';
            $product_height_per_parcel = isset($parameters['product_height_per_parcel_' . $i]) ? $parameters['product_height_per_parcel_' . $i] : '';
            if ($i === 1) {
                $globalParcel[0] = $product_length_per_parcel !== '' ? (int)$product_length_per_parcel : 50;
                $globalParcel[1] = $product_width_per_parcel !== '' ? (int)$product_width_per_parcel : 50;
                $globalParcel[2] = $product_height_per_parcel !== '' ? (int)$product_height_per_parcel : 50;
                rsort($globalParcel);
                $globalParcels[0] = $globalParcel;
            } else {
                $skip = false;
                if ($product_length_per_parcel === '') {
                    $skip = true;
                }
                if ($product_width_per_parcel === '') {
                    $skip = true;
                }
                if ($product_height_per_parcel === '') {
                    $skip = true;
                }
                if (!$skip) {
                    $globalParcel[0] = (int)$product_length_per_parcel;
                    $globalParcel[1] = (int)$product_width_per_parcel;
                    $globalParcel[2] = (int)$product_height_per_parcel;
                    rsort($globalParcel);
                    $globalParcels[$i - 1] = $globalParcel;
                }
            }
        }
        $globalParcelCount = count($globalParcels);

        // Order the global parcels by largest dimension ascending order
        if (count($globalParcels) > 1) {
            usort(
                $globalParcels,
                function ($a, $b){
                    if ($a[0] === $b[0]) {
                        return 0;
                    }
                    return ($a[0] < $b[0]) ? -1 : 1;
                }
            );
        }

        /**
         * Check for any single-packaging product items
         * They will be packaged individually using their own dimensions
         * Global parcels don't apply
         */
        $singleItems = [];
        foreach ($items as $item) {
            $item_variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
            $item_product_id   = isset($item['product_id']) ? $item['product_id'] : 0;
            if ($item_variation_id !== 0) {
                $product = new WC_Product_Variation($item_variation_id);
            } else {
                $product = new WC_Product($item_product_id);
            }
            if ($this->isSingleProductItem($product)) {
                $singleItems[] = $item;
                if (isset($items[$item['key']])) {
                    unset($items[$item['key']]);
                } else {
                    $cnt = 0;
                    foreach ($items as $im) {
                        if ($im['key'] === $item['key']) {
                            unset($items[$cnt]);
                        }
                        $cnt++;
                    }
                }
            }
        }

        /**
         * Items that don't fit into any of the defined parcel sizes
         * are passed as individual items with their own dimension and mass
         *
         * Now check if there are items that don't fit into any box
         */
        $i            = 0;
        $tooBigItems  = [];
        $fittingItems = [];
        foreach ($items as $item) {
            $item_key = isset($item['key']) ? $item['key'] : 0;
            $fits     = $this->doesFitGlobalParcels($item, $globalParcels);
            if (!$fits['fits']) {
                $tooBigItems[] = $item_key;
            } else {
                $fittingItems[] = ['key' => $item_key, 'index' => $fits['fitsIndex']];
            }
        }

        // Order the fitting items with the biggest dimension first
        usort(
            $fittingItems,
            function ($a, $b) use ($items, $fittingItems){
                if (isset($items[$a['key']])) {
                    $itema = $items[$a['key']];
                } else {
                    foreach ($items as $item) {
                        if ($item['key'] === $a['key']) {
                            $itema = $item;
                        }
                    }
                }
                if (isset($items[$b['key']])) {
                    $itemb = $items[$b['key']];
                } else {
                    foreach ($items as $item) {
                        if ($item['key'] === $b['key']) {
                            $itemb = $item;
                        }
                    }
                }
                $item_variation_id_a = isset($itema['variation_id']) ? $itema['variation_id'] : 0;
                $item_product_id_a   = isset($itema['product_id']) ? $itema['product_id'] : 0;
                if ($item_variation_id_a === 0) {
                    $producta = new WC_Product($item_product_id_a);
                } else {
                    $producta = new WC_Product_Variation(($item_variation_id_a));
                }
                $item_variation_id_b = isset($itemb['variation_id']) ? $itemb['variation_id'] : 0;
                $item_product_id_b   = isset($itemb['product_id']) ? $itemb['product_id'] : 0;
                if ($item_variation_id_b === 0) {
                    $productb = new WC_Product($item_product_id_b);
                } else {
                    $productb = new WC_Product_Variation(($item_variation_id_b));
                }
                if ($producta->has_dimensions()) {
                    $producta_size = max(
                        (int)$producta->get_length(),
                        (int)$producta->get_width(),
                        (int)$producta->get_height()
                    );
                } else {
					$producta_size = 1;
                }
                if ($productb->has_dimensions()) {
                    $productb_size = max(
                        (int)$productb->get_length(),
                        (int)$productb->get_width(),
                        (int)$productb->get_height()
                    );
                } else {
					$productb_size = 1;
                }
                if ($producta_size === $productb_size) {
                    return 0;
                }
                return ($producta_size < $productb_size) ? 1 : -1;
            }
        );

        // Handle the single parcel items first
        $j = 0;
        foreach ($singleItems as $singleItem) {
            $j++;
            /** Items format differs when multi-vendor plugin is enabled */
            if (isset($items[$singleItem['key']])) {
                $item = $items[$singleItem['key']];
            } else {
                foreach ($singleItems as $itm) {
                    if ($itm['key'] === $singleItem['key']) {
                        $item = $itm;
                    }
                }
            }
            $item_variation_id = isset($singleItem['variation_id']) ? $singleItem['variation_id'] : 0;
            $item_product_id   = isset($singleItem['product_id']) ? $singleItem['product_id'] : 0;
            if ($item_variation_id === 0) {
                $product = new WC_Product($item_product_id);
            } else {
                $product = new WC_Product_Variation(($item_variation_id));
            }
            $slug  = get_post($item_product_id)->post_title;
            $entry = [];
            if ($product->has_dimensions()) {
                $dim['dim1'] = (int)$product->get_width();
                $dim['dim2'] = (int)$product->get_height();
                $dim['dim3'] = (int)$product->get_length();
                sort($dim);
                $entry['dim1'] = $dim[0];
                $entry['dim2'] = $dim[1];
                $entry['dim3'] = $dim[2];
            } else {
                $entry['dim1'] = 1;
                $entry['dim2'] = 1;
                $entry['dim3'] = 1;
            }
            if ($product->has_weight()) {
                $entry['actmass'] = $product->get_weight();
            } else {
                $entry['actmass'] = 1.0;
            }

            for ($i = 0; $i < $item['quantity']; $i++) {
                $entry['item']        = $j;
                $entry['description'] = !$waybillDescriptionOverride ? $slug : 'Item';
                $entry['pieces']      = 1;
                $r1[]                 = $entry;
                $j++;
            }
            $j--;
        }

        // Handle the non-fitting items next
        // Single pack sizes
        foreach ($tooBigItems as $tooBigItem) {
            $j++;
            /** Items format differs when multi-vendor plugin is enabled */
            if (isset($items[$tooBigItem])) {
                $item = $items[$tooBigItem];
            } else {
                foreach ($items as $itm) {
                    if ($itm['key'] === $tooBigItem) {
                        $item = $itm;
                    }
                }
            }
            $item_variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
            $item_product_id   = isset($item['product_id']) ? $item['product_id'] : 0;
            if ($item_variation_id === 0) {
                $product = new WC_Product($item_product_id);
            } else {
                $product = new WC_Product_Variation(($item_variation_id));
            }
            $slug                 = get_post($item_product_id)->post_title;
            $entry                = [];
            $entry['item']        = $j;
            $entry['description'] = !$waybillDescriptionOverride ? $slug : 'Item';
            $entry['pieces']      = $item['quantity'];

            if ($product->has_dimensions()) {
                $dim['dim1'] = (int)$product->get_width();
                $dim['dim2'] = (int)$product->get_height();
                $dim['dim3'] = (int)$product->get_length();
                sort($dim);
                $entry['dim1'] = $dim[0];
                $entry['dim2'] = $dim[1];
                $entry['dim3'] = $dim[2];
            } else {
                $entry['dim1'] = 1;
                $entry['dim2'] = 1;
                $entry['dim3'] = 1;
            }
            if ($product->has_weight()) {
                $entry['actmass'] = $item['quantity'] * $product->get_weight();
            } else {
                $entry['actmass'] = 1.0;
            }
            $r1[] = $entry;
        }

		$this->poolIfPossible( $fittingItems, $items );

        /** Now the fitting items
         * We have to fit them into parcels
         * The idea is to minimise the total number of parcels - cf Talent 2020-09-09
         *
         */
        if (count($fittingItems) === 1) {
            // Handle with existing code which works
            foreach ($fittingItems as $fittingItem) {
                /** Items format differs when multi-vendor plugin is enabled */
                if (isset($items[$fittingItem['key']])) {
                    $item = $items[$fittingItem['key']];
                } else {
                    foreach ($items as $itm) {
                        if (isset($itm['key']) && $itm['key'] === $fittingItem['key']) {
                            $item = $itm;
                        }
                    }
                }
                $item_variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
                $item_product_id   = isset($item['product_id']) ? $item['product_id'] : 0;
                if ($item_variation_id === 0) {
                    $product = new WC_Product($item_product_id);
                } else {
                    $product = new WC_Product_Variation(($item_variation_id));
                }
                if ($product->has_dimensions()) {
                    // Fit them into boxes
                    $pdims = [$product->get_length(), $product->get_width(), $product->get_height()];
				} else {
					$pdims = [1,1,1];
				}

                    // Calculate how many items will fit into a box
                    $maxItems        = 0;
                    $initialBoxIndex = $fittingItem['index'];
                    $bestFit         = false;
                    while (!$bestFit) {
                        $maxItems = $this->getMaxPackingConfiguration($globalParcels[$initialBoxIndex], $pdims);

                        $nboxes = (int)ceil($item['quantity'] / $maxItems);
                        if ($nboxes > 1 && $initialBoxIndex < (count($globalParcels) - 1)) {
                            $initialBoxIndex++;
                        } else {
                            $bestFit = true;
                        }
                    }

                    for ($box = 1; $box <= $nboxes; $box++) {
                        $j++;
                        $entry = [];
                        if ($box !== $nboxes) {
                            $entry['pieces'] = $maxItems;
                        } else {
                            $entry['pieces'] = $item['quantity'] - ($box - 1) * $maxItems;
                        }
                        $slug                 = get_post($item_product_id)->post_title;
                        $entry['item']        = $j;
                        $entry['description'] = !$waybillDescriptionOverride ? $slug : 'Item';
                        if ($product->has_weight()) {
                            $entry['actmass'] = $entry['pieces'] * $product->get_weight();
                        } else {
                            $entry['actmass'] = 1.0;
                        }
                        $entry['pieces'] = 1; // Each box counts as one piece
                        $entry['dim1']   = $globalParcels[$initialBoxIndex][0];
                        $entry['dim2']   = $globalParcels[$initialBoxIndex][1];
                        $entry['dim3']   = $globalParcels[$initialBoxIndex][2];
                        $r1[]            = $entry;
                    }
            }
        } elseif(count($fittingItems) > 1) {
            // Have more than one size items to try and pack
            // Start with the smallest box that will fit all products
            // and the largest product
            $initialBoxIndex = 0;
            unset($item);
            foreach ($fittingItems as $fittingItem) {
                if ($fittingItem['index'] > $initialBoxIndex) {
                    $initialBoxIndex = $fittingItem['index'];
                }
            }

            $bestFit = false;
            $k       = $j;

            while (!$bestFit) {
                $itemIndex    = 0;
                $anyItemsLeft = true;
                $nboxes       = 1;
                $r2           = [];
                $j            = $k;
                $j++;
                $entry                = [];
                $entry['description'] = '';
                $entry['actmass']     = 0;
                $boxIsFull            = false;
                $boxRemaining         = $globalParcels[$initialBoxIndex];
                while ($anyItemsLeft && $itemIndex !== count($fittingItems)) {
                    if ($boxIsFull) {
                        $nboxes++;
                        $j++;
                        $entry                = [];
                        $entry['actmass']     = 0;
                        $entry['description'] = '';
                        $boxIsFull            = false;
                        $boxRemaining         = $globalParcels[$initialBoxIndex];
                    }
                    /** Items format differs when multi-vendor plugin is enabled */
                    if (!isset($item)) {
                        if (isset($items[$fittingItems[$itemIndex]['key']])) {
                            $item = $items[$fittingItems[$itemIndex]['key']];
                        } else {
                            foreach ($items as $itm) {
                                if (isset($itm['key']) && $itm['key'] === $fittingItems[$itemIndex]['key']) {
                                    $item = $itm;
                                }
                            }
                        }
                    }
                    $item_variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
                    $item_product_id   = isset($item['product_id']) ? $item['product_id'] : 0;
                    if ($item_variation_id === 0) {
                        $product = new WC_Product($item_product_id);
                    } else {
                        $product = new WC_Product_Variation(($item_variation_id));
                    }
                    if ($product->has_dimensions()) {
                        // Calculate how many can be added
                        $pdims = [$product->get_length(), $product->get_width(), $product->get_height()];
					} else {
						$pdims = [1,1,1];
					}
                        // Calculate max that can be filled
                        $maxItems             = $this->getMaxPackingConfiguration($boxRemaining, $pdims);
                        $slug                 = get_post($item_product_id)->post_title;
                        $entry['item']        = $j;
                        $entry['description'] .= '_' . !$waybillDescriptionOverride ? $slug : 'Item';
                        $entry['pieces']      = 1;
                        $entry['dim1']        = $globalParcels[$initialBoxIndex][0];
                        $entry['dim2']        = $globalParcels[$initialBoxIndex][1];
                        $entry['dim3']        = $globalParcels[$initialBoxIndex][2];
                        if ($maxItems >= $item['quantity']) {
                            // Put them all in
                            if ($product->has_weight()) {
                                $entry['actmass'] += $item['quantity'] * $product->get_weight();
                            } else {
                                $entry['actmass'] += 1.0;
                            }
                            $itemIndex++;
                            if ($itemIndex == count($fittingItems)) {
                                $anyItemsLeft = false;
                            }
                            if ($item['quantity'] === $maxItems) {
                                $boxIsFull = true;
                            }
                            // Calculate the remaining box content
                            $used            = $this->getActualPackingConfiguration(
                                $boxRemaining,
                                $pdims,
                                $item['quantity']
                            );
                            $boxRemaining[2] -= $used;
                            unset($item);
                        } else {
                            // Fill the box and calculate remainder
                            if ($product->has_weight()) {
                                $entry['actmass'] += $maxItems * $product->get_weight();
                            } else {
                                $entry['actmass'] += 1.0;
                            }
                            $boxIsFull        = true;
                            $r2[]             = $entry;
                            $item['quantity'] -= $maxItems;
                            continue;
                        }
                }
                $r2[] = $entry;
                if ($nboxes === 1 || ($nboxes > 1 && $initialBoxIndex == count($globalParcels) - 1)) {
                    $bestFit = true;
                }
                $initialBoxIndex++;
            }
        }

        foreach ($r2 as $item) {
            $r1[] = $item;
        }
        return array_values($r1);
    }

	/**
	 * Will attempt to pool items of same dimensions to produce
	 * better packing calculations
	 *
	 * Parameters are passed by reference, so modified in the function
	 * @param $fittingItems
	 * @param $items
	 */
	private function poolIfPossible( &$fittingItems, &$items ) {
		$pooledItems = [];
		$dimensions  = [];

		foreach ( $fittingItems as $fittingItem ) {
			if ( isset( $items[ $fittingItem['key'] ] ) ) {
				$item = $items[ $fittingItem['key'] ];
			} else {
				foreach ( $items as $itm ) {
					if ( isset( $itm['key'] ) && $itm['key'] === $fittingItem['key'] ) {
						$item = $itm;
					}
				}
			}

			$item_variation_id = isset( $item['variation_id'] ) ? $item['variation_id'] : 0;
			$item_product_id   = isset( $item['product_id'] ) ? $item['product_id'] : 0;
			if ( $item_variation_id === 0 ) {
				$product = new WC_Product( $item_product_id );
			} else {
				$product = new WC_Product_Variation( ( $item_variation_id ) );
			}

			if ( $product->has_dimensions() ) {
				$pdims = [ (int) $product->get_length(), (int) $product->get_width(), (int) $product->get_height() ];
			} else {
				$pdims = [1,1,1];
			}
				sort( $pdims, SORT_NUMERIC );
				$pvol  = (int) ( $pdims[0] * $pdims[1] * $pdims[2] );
				$pmass = 1.0;
				if ( $product->has_weight() ) {
					$pmass = $product->get_weight();
				}
				$pcount       = $item['quantity'];
				$dimensions[] = [
					'dim'   => $pdims,
					'vol'   => $pvol,
					'count' => $pcount,
					'mass'  => $pmass,
					'key'   => $fittingItem['key'],
				];
		}

		$pool = [];
		$k    = count( $dimensions );
		while ( $k > 0 ) {
			$match = false;
			for ( $i = 0; $i < count( $dimensions ); $i ++ ) {
				if ( count( $pool ) > 0 ) {
					for ( $p = 0; $p < count( $pool ); $p ++ ) {
						if (
							$dimensions[ $i ]['dim'][0] === $pool[ $p ]['dim'][0]
							&& $dimensions[ $i ]['dim'][1] === $pool[ $p ]['dim'][1]
							&& $dimensions[ $i ]['dim'][2] === $pool[ $p ]['dim'][2]
						) {
							$match    = true;
							$newCount = $pool[ $p ]['count'] + $dimensions[ $i ]['count'];
							$newMass  = $pool[ $p ]['mass'] * $pool[ $p ]['count'] + $dimensions[ $i ]['count'] * $dimensions[ $i ]['mass'];
							$newMass  /= $newCount;

							$pool[ $p ]['count'] = $newCount;
							$pool[ $p ]['mass']  = $newMass;

							array_splice($dimensions, $i, 1);

							$k --;
							break;
						}
					}
					if ( ! $match && $k === 1 ) {
						$k --;
					}
				}
				for ( $j = 0; $j < count( $dimensions ); $j ++ ) {
					if ( $i < $j ) {
						if ( $dimensions[ $i ]['vol'] === $dimensions[ $j ]['vol'] ) {
							if (
								$dimensions[ $i ]['dim'][0] === $dimensions[ $j ]['dim'][0]
								&& $dimensions[ $i ]['dim'][1] === $dimensions[ $j ]['dim'][1]
								&& $dimensions[ $i ]['dim'][2] === $dimensions[ $j ]['dim'][2]
							) {
								$match = true;
								if ( count( $pool ) === 0 ) {
									$poolItem = [
										'dim'   => $dimensions[ $i ]['dim'],
										'vol'   => $dimensions[ $i ]['vol'],
										'count' => $dimensions[ $i ]['count'] + $dimensions[ $j ]['count'],
										'mass'  => ( $dimensions[ $i ]['mass'] * $dimensions[ $i ]['count'] + $dimensions[ $j ]['mass'] * $dimensions[ $j ]['count'] ) / ( $dimensions[ $i ]['count'] + $dimensions[ $j ]['count'] ),
										'key'   => $dimensions[ $i ]['key']
									];
									$pool[]   = $poolItem;
									$k        -= 2;
									array_splice($dimensions, $i, 1);
									array_splice($dimensions, $j - 1, 1);
									break 2;
								} else {
									for ( $p = 0; $p < count( $pool ); $p ++ ) {
										if (
											$dimensions[ $i ]['dim'][0] === $pool[ $p ]['dim'][0]
											&& $dimensions[ $i ]['dim'][1] === $pool[ $p ]['dim'][1]
											&& $dimensions[ $i ]['dim'][2] === $pool[ $p ]['dim'][2]
										) {
											$match               = true;
											$newCount            = $pool[ $p ] + $dimensions[ $i ]['count'] + $dimensions[ $j ]['count'];
											$newMass             = $pool[ $p ]['mass'] * $pool[ $p ]['count'] + $dimensions[ $i ]['count'] * $dimensions[ $i ]['mass'] + $dimensions[ $i ]['count'] * $dimensions[ $i ]['mass'];
											$newMass             /= $newCount;
											$pool[ $p ]['count'] = $newCount;
											$pool[ $p ]['mass']  = $newMass;

											$k -= 2;
											break 2;
										}
									}
								}
							}
						}
					}
				}
			}
			if ( ! $match && count( $dimensions ) > 0 ) {
				foreach ( $dimensions as $dimension ) {
					$pooledItems[] = $dimension;
				}
				$k = 0;
			}
		}
		foreach ( $pool as $item ) {
			$pooledItems[] = $item;
		}

		$pooledCounts[] = array_map( function ( $pooledItem ) {
			return $pooledItem['count'];
		},
			$pooledItems );
		$pooledCounts   = $pooledCounts[0];

		$pooledKeys[] = array_map( function ( $pooledItem ) {
			return $pooledItem['key'];
		},
			$pooledItems );
		$pooledKeys   = $pooledKeys[0];

		$pooled = array_combine( $pooledKeys, $pooledCounts );

		$itemsKeys[] = array_map( function ( $item ) {
			return $item['key'];
		},
			$items );
		$itemsKeys   = $itemsKeys[0];

		$fittingItemKeys[] = array_map( function ( $fit ) {
			return $fit['key'];
		},
			$fittingItems );
		$fittingItemKeys   = $fittingItemKeys[0];

		foreach ( $items as $k => $item ) {
			if ( in_array( $item['key'], $fittingItemKeys ) ) {
				if ( in_array( $item['key'], $pooledKeys ) ) {
					$items[ $k ]['quantity'] = $pooled[ $k ];
				} else {
					unset( $items[ $k ] );
				}
			}
		}

		foreach ( $fittingItems as $k => $fittingItem ) {
			if ( ! in_array( $fittingItem['key'], $pooledKeys ) ) {
				unset($fittingItems[$k]);
			}
		}
		$fittingItems = array_values($fittingItems);
	}

    /**
     * @param $product
     * @return bool
     */
    private function isSingleProductItem($product)
    {
        $psp = get_post_meta($product->get_id(), 'product_single_parcel');
        if (is_array($psp) && count($psp) > 0) {
            return get_post_meta($product->get_id(), 'product_single_parcel')[0] === 'on';
        }
        return false;
    }

    /**
     * @param $parcel
     * @param $package
     * @param $count
     * @return float|int|mixed
     */
    private function getActualPackingConfiguration($parcel, $package, $count)
    {
        $boxPermutations = [
            [0, 1, 2],
            [0, 2, 1],
            [1, 0, 2],
            [1, 2, 0],
            [2, 1, 0],
            [2, 0, 1]
        ];

        $usedHeight = $parcel[2];
        foreach ($boxPermutations as $permutation) {
            $nl = (int)($parcel[0] / $package[$permutation[0]]);
            $nw = (int)($parcel[1] / $package[$permutation[1]]);
            $na = $nl * $nw;
            if ($na !== 0) {
                $h = ceil($count / ($nl * $nw)) * $package[$permutation[2]];
                if ($h < $usedHeight) {
                    $usedHeight = $h;
                }
            }
        }

        return $usedHeight;
    }

    /**
     * @param $parcel
     * @param $package
     * @return mixed
     */
    private function getMaxPackingConfiguration($parcel, $package)
    {
        $boxPermutations = [
            [0, 1, 2],
            [0, 2, 1],
            [1, 0, 2],
            [1, 2, 0],
            [2, 1, 0],
            [2, 0, 1]
        ];

        $maxItems = 0;
        foreach ($boxPermutations as $key => $permutation) {
            $boxItems = (int)($parcel[0] / $package[$permutation[0]]);
            $boxItems *= (int)($parcel[1] / $package[$permutation[1]]);
            $boxItems *= (int)($parcel[2] / $package[$permutation[2]]);
            $maxItems = max($maxItems, $boxItems);
        }
        return $maxItems;
    }

    /**
     * @param $item
     * @param $globalParcels
     * @return array
     */
    private function doesFitGlobalParcels($item, $globalParcels)
    {
        $globalParcelIndex = 0;
        foreach ($globalParcels as $globalParcel) {
            $fits = $this->doesFitParcel($item, $globalParcel);
            if ($fits) {
                break;
            }
            $globalParcelIndex++;
        }
        return ['fits' => $fits, 'fitsIndex' => $globalParcelIndex];
    }

    /**
     * @param $item
     * @param $parcel
     * @return bool
     */
    private function doesFitParcel($item, $parcel)
    {
        rsort($parcel);
        $item_variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
        $item_product_id   = isset($item['product_id']) ? $item['product_id'] : 0;
        if ($item_variation_id !== 0) {
            $product = new WC_Product_Variation($item_variation_id);
        } else {
            $product = new WC_Product($item_product_id);
        }
        if ($product->has_dimensions()) {
            $productDims    = [];
            $productDims[0] = $product->get_length();
            $productDims[1] = $product->get_width();
            $productDims[2] = $product->get_height();
		} else {
			$productDims    = [];
			$productDims[0] = 1;
			$productDims[1] = 1;
			$productDims[2] = 1;
		}
            rsort($productDims);
            $fits = false;
            if (
                $productDims[0] <= $parcel[0]
                && $productDims[1] <= $parcel[1]
                && $productDims[2] <= $parcel[2]
            ) {
                $fits = true;
            }
        return $fits;
    }

    /**
     * @param array $package
     * @param array $parameters
     *
     * @return array
     */
    public
    function getQuotePayload(
        $package,
        $parameters
    ){
        /** @var TCG_Plugin $TCG_Plugin */
        global $TCG_Plugin;
        $result                   = [];
        $customShippingProperties = $TCG_Plugin->getShippingCustomProperties();
        if (!empty($parameters) && !empty($customShippingProperties['tcg_place_id']) && !empty($customShippingProperties['tcg_place_label'])) {
            $originPayload      = $this->getOriginPayload($parameters);
            $destinationPayload = $this->getDestinationPayloadForQuote($package);
            $insurancePayload   = $this->getInsurancePayloadForQuote();
            $detailsPayload     = array_merge(
                $originPayload,
                ['reference' => $destinationPayload['destpers'],],
                $destinationPayload,
                $insurancePayload
            );
            $result['details']  = $detailsPayload;
            $contentsPayload    = $this->getContentsPayload($parameters, $package['contents']);
            $result['contents'] = $contentsPayload;
        }

        return $result;
    }

    /**
     * @param WC_Order $order
     * @param array $shippingItem
     * @param array $parameters
     *
     * @return array
     */
    public
    function getCollectionPayload(
        $order,
        $shippingItem,
        $parameters
    ){
        $result = [];
        if (!empty($order) && !empty($parameters)) {
            $originPayload                    = $this->getOriginPayload($parameters);
            $originPayload['notes']           = $order->get_customer_note();
            $destinationPayload               = $this->getDestinationPayloadForCollection($order);
            $insurancePayload                 = $this->getInsurancePayloadForCollection($order);
            $detailsPayload                   = array_merge($originPayload, $destinationPayload, $insurancePayload);
            $detailsPayload['service']        = $this->getServiceIdentifierFromShippingItem($shippingItem);
            $detailsPayload['collectiondate'] = current_time('d.m.Y');
            $detailsPayload['starttime']      = current_time('H:i:s');
            $detailsPayload['endtime']        = '18:00:00';
            $result['details']                = $detailsPayload;
            $orderItems                       = $order->get_items('line_item');
            $contentsPayload                  = $this->getContentsPayload($parameters, $orderItems);
            $result['contents']               = $contentsPayload;
        }

        return $result;
    }

    /**
     * @param array $shippingItem
     *
     * @return mixed
     */
    private
    function getServiceIdentifierFromShippingItem(
        $shippingItem
    ){
        $method      = $shippingItem['method_id'];
        $methodParts = explode(':', $method);

        return $methodParts[1];
    }
}
